﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI.WebControls;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager;
using VeteransAffairs.Registries.BusinessManager.Utilities;
using System.Web.Security.AntiXss;

public partial class LabResultDetails : BasePage
{
    private readonly LabResultsManager _labResultsManager;
    private readonly UserAccountManager _userAccountManager;
    private readonly LabOrderManager _labOrderManager;
    private readonly WorkflowCaseManager _workflowManager;

    private WKF_CASE_ACTIVITY _activity;
    private WKF_CASE_ACTIVITY _order;
    private WKF_CASE_ACTIVITY _labKit;
    private WKF_CASE _wCase;
    private STD_WKFACTIVITYST _activitySt;

    private static IReadOnlyList<string> _navigationSources = new List<string>
    {
        "/LabResults.aspx",
        "/PatientWorkflows.aspx"
    };

    public LabResultDetails()
    {
        _labResultsManager = new LabResultsManager();
        _labOrderManager = new LabOrderManager();
        _workflowManager = new WorkflowCaseManager();
        _userAccountManager = (UserAccountManager)HttpContext.Current.User;

        //Manager Save Notification Event Handler
        _labResultsManager.BOSaveSuccess += manager_BOSaveSuccess;
        _labOrderManager.BOSaveSuccess += manager_BOSaveSuccess;

        NavigationSources = _navigationSources;
    }

    protected void Page_Load(object sender, EventArgs e)
    {
        if (!IsPostBack)
        {

            int WKFCaseActivityId = Convert.ToInt32(AntiXssEncoder.HtmlEncode(Request.QueryString["ActivityId"], false));

            BindData(WKFCaseActivityId);
            //TODO - Move breadcrumb logic to menubuilder and master page. Added here to avoid error out in Referral pages.

            SetReferringPage();
        }

        //Only TEFSC ADMINISTRATOR or EFR_TEFSC_COORDINATOR can accept Lab Order Results.
        if (!_userAccountManager.GetRoles().Contains((int)RoleCode.EFR_TEFSC_ADMINISTRATOR)
            && !_userAccountManager.GetRoles().Contains((int)RoleCode.EFR_TEFSC_COORDINATOR))
        {
            btnAccept.Enabled = false;
        }

        //Per bug 612: Only TEFSC_COORDINATOR can reject Lab Order Results.
        if (!_userAccountManager.GetRoles().Contains((int)RoleCode.EFR_TEFSC_COORDINATOR))
        {
            btnReject.Enabled = false;
        }
    }

    private void Page_PreRender(object sender, EventArgs e)
    {
        DisableRequiredValidatorForResultsWhenUpdating();
    }

    private void DisableRequiredValidatorForResultsWhenUpdating()
    {
        if (ViewState["EditMode"] != null)
        {
            if (AnalyteList.EditItemIndex >= 0)
            {
                DataListItem listItem = AnalyteList.Items[AnalyteList.EditItemIndex];
                if (listItem != null)
                {
                    RequiredFieldValidator validator = (RequiredFieldValidator)listItem.FindControl("AnalysisResultBoxRequiredValidator");
                    if (validator != null)
                    {
                        if (Convert.ToBoolean(ViewState["EditMode"]))
                        {
                            validator.Enabled = false;
                        }
                        else
                        {
                            validator.Enabled = true;
                        }
                    }
                }
            }
        }
    }

    #region DataList events 
    protected void AnalyteList_EditCommand(object sender, DataListCommandEventArgs e)
    {
        Deserialize();
        AnalyteList.EditItemIndex = e.Item.ItemIndex;
        AnalyteList.ShowFooter = false;
        ViewState["EditMode"] = true;

        BindDataList();

    }

    protected void AnalyteList_UpdateCommand(object sender, DataListCommandEventArgs e)
    {
        Deserialize();
        AnalyteList.EditItemIndex = -1;
        AnalyteList.ShowFooter = true;

        int index = e.Item.ItemIndex;
        AnalyteList.SelectedIndex = index;
        ViewState["EditMode"] = false;
        int id = int.Parse(AnalyteList.SelectedValue.ToString());
        UpdateCreateDataListAnalyte(e.Item, id, false);
        BindDataList();

    }

    protected void AnalyteList_CancelCommand(object sender, DataListCommandEventArgs e)
    {
        Deserialize();
        AnalyteList.EditItemIndex = -1;
        AnalyteList.ShowFooter = true;
        ViewState["EditMode"] = false;
        BindDataList();
    }

    protected void AnalyteList_DeleteCommand(object sender, DataListCommandEventArgs e)
    {
        Deserialize();
        int index = e.Item.ItemIndex;
        AnalyteList.SelectedIndex = index;

        int id = int.Parse(AnalyteList.SelectedValue.ToString());

        DeleteAnalyte(id);

        AnalyteList.ShowFooter = true;
        BindDataList();


    }
    //have to load analyte in this event since edit template already loaded at this point
    protected void AnalyteList_DataBound(object sender, DataListItemEventArgs e)
    {

        if (e.Item.ItemType == ListItemType.EditItem && bool.Parse(ViewState["EditMode"].ToString()) == true)
        {
            //AnalyteList.DataKeys(e.Item.ItemIndex)
            int index = e.Item.ItemIndex;
            AnalyteList.SelectedIndex = index;

            int id = int.Parse(AnalyteList.SelectedValue.ToString());
            LoadDataListAnalyte(e.Item, id);


        }


        Deserialize();

        if (_activity.STD_WKFACTIVITYSTS_ID == (int)WorkflowCaseActivityStatus.Accepted)
        {

            if (e.Item.ItemType == ListItemType.Item || e.Item.ItemType == ListItemType.AlternatingItem
                || e.Item.ItemType == ListItemType.SelectedItem)
            {
                Button edit = e.Item.FindControl("ButtonEdit") as Button;

                if (edit != null)
                    edit.Enabled = false;

                Button delete = e.Item.FindControl("ButtonDelete") as Button;

                if (delete != null)
                    delete.Enabled = false;

            }

            else if (e.Item.ItemType == ListItemType.Footer)
            {

                DisableAllControls(e.Item);

            }

        }

    }

    protected void AnalyteList_ItemCommand(object sender, DataListCommandEventArgs e)
    {

        int id = 0;
        if (e.CommandName.Equals("Insert"))
        {
            Deserialize();
            UpdateCreateDataListAnalyte(e.Item, id, true);
            BindDataList();
            //set the flag for dropdown focus
            dataListViewAddModeFlag.Value = "1";
            ClearAllControls(e.Item);
            //(this.Master as RegistryMaster).SetDirtyFlag(false); 
        }

    }

    protected void AnalyteList_PreRender(object sender, EventArgs e)
    {
        DataList AnalyteDL = (DataList)sender;
        if (AnalyteDL.Items.Count == 0)
        {
            //AnalyteDL.ShowHeader = true; 
            AnalyteDL.ShowFooter = true;
        }



    }

    #endregion 

    private void ClearAllControls(DataListItem item)
    {

        List<TextBox> textBoxes = item.GetAllControls().OfType<TextBox>().ToList();

        textBoxes.ForEach(elem => elem.Text = String.Empty);

        List<DropDownList> lists = item.GetAllControls().OfType<DropDownList>().ToList();

        lists.ForEach(elem => elem.SelectedIndex = -1);

        List<CheckBox> checks = item.GetAllControls().OfType<CheckBox>().ToList();

        checks.ForEach(elem => elem.Checked = false);

    }


    private void DisableAllControls(DataListItem item)
    {

        List<TextBox> textBoxes = item.GetAllControls().OfType<TextBox>().ToList();

        textBoxes.ForEach(elem => elem.Enabled = false);

        List<DropDownList> lists = item.GetAllControls().OfType<DropDownList>().ToList();

        lists.ForEach(elem => elem.Enabled = false);

        List<CheckBox> checks = item.GetAllControls().OfType<CheckBox>().ToList();

        checks.ForEach(elem => elem.Enabled = false);

        List<Button> buttons = item.GetAllControls().OfType<Button>().ToList();

        buttons.ForEach(elem => elem.Enabled = false);

    }

    private void GetRefreshLabResults(int activityId)
    {
        _wCase = _labResultsManager.GetLabResultDetails(activityId);

        //TODO - Check the default value
        _activity = _wCase.WKF_CASE_ACTIVITies[0];

        ViewState["WCase"] = _wCase.Serialize();


        _order = _labOrderManager.GetLabOrderByCaseId(_wCase.WKF_CASE_ID);
        ViewState["Order"] = _order.Serialize();

        _labKit = _workflowManager.SelectByActivityType(_wCase.WKF_CASE_ID, "LabKit");

        ViewState["LabKit"] = _labKit.Serialize();

        if (_order != null)
        {
            LabOrderLink.Visible = true;
            LabOrderLink.NavigateUrl = "LabOrder.aspx?activityId=" + _order.WKF_CASE_ACTIVITY_ID;

        }

    }

    private void BindData(int wkfCaseActivityId)
    {
        GetRefreshLabResults(wkfCaseActivityId);

        PatientLabInfo Info = new PatientLabInfo();

        if (_activity != null)
        {
            Info.Name = _activity.WKF_CASE.REFERRAL.PATIENT.FullName;
            Info.ID = _activity.WKF_CASE.REFERRAL.PATIENT.PATIENT_ID.ToString();
            Info.ICN = _activity.WKF_CASE.REFERRAL.PATIENT.PATIENT_ICN;
            Info.Snum = _activity.WKF_CASE.REFERRAL.PATIENT.Snum;
            Info.BirthDate = ConvertDateToString(_activity.WKF_CASE.REFERRAL.PATIENT.BIRTH_DATE);
            Info.Gender = _activity.WKF_CASE.REFERRAL.PATIENT.STD_GENDER.NAME;
            Info.LabContact = _activity.CONTACT_NAME;
            Info.PostalCode = _activity.WKF_CASE.REFERRAL.PATIENT.POSTAL_CODE;
            Info.WorkFlowCaseId = _activity.WKF_CASE_ID.ToString();
            _activitySt = _activity.STD_WKFACTIVITYST;
            ViewState["OrderType"] = _activitySt.ID;
            switch (_activitySt.ID)
            {

                case (int)WorkflowCaseActivityStatus.New:

                    btnSave.Enabled = btnAccept.Enabled = btnReject.Enabled = true;
                    btnSave.Text = "Save";
                    break;

                case (int)WorkflowCaseActivityStatus.InProcess:

                    btnSave.Enabled = btnAccept.Enabled = btnReject.Enabled = true;
                    btnSave.Text = "Save";
                    break;

                case (int)WorkflowCaseActivityStatus.Accepted:

                    btnSave.Enabled = btnReject.Enabled = btnAccept.Enabled = false;
                    btnSave.Text = "Save";
                    break;

                case (int)WorkflowCaseActivityStatus.Voided:

                    btnSave.Enabled = btnAccept.Enabled = true;
                    btnReject.Enabled = false;
                    btnSave.Text = "Un-Void";
                    break;

                default:

                    break;

            }
            Helpers.UnacceptLabResults(_activitySt.ID, _activity.WKF_CASE_ID, btnAccept, _userAccountManager);

            if (_activity.SPECIMEN_ANALYSIs.Count() > 0)
            {

                Info.Accession = _activity.SPECIMEN_ANALYSIs[0].LAB_UID_NUMBER;

                Info.SpecColnDate = ConvertDateToString(_activity.SPECIMEN_ANALYSIs[0].SPECIMEN_COLLECTION_DATE);
                txtElapsedTime.Text = _activity.SPECIMEN_ANALYSIs[0].TIME_ELAPSED_NUMBER;
                txtTotalVolume.Text = _activity.SPECIMEN_ANALYSIs[0].TOTAL_VOLUME_NUMBER;
                txtVolOfAnalyteTests.Text = _activity.SPECIMEN_ANALYSIs[0].ANALYTE_VOLUME_NUMBER;
                //Populate Analyte details
                Info.ReportDate = ConvertDateToString(_activity.SPECIMEN_ANALYSIs.First().LAB_REPORT_DATE);
                Info.LabName = _activity.SPECIMEN_ANALYSIs[0].LAB_NAME;
            }
            if (Info.LabName == null)
                Info.LabName = "Baltimore VA lab";


            if (_order == null)
                _order = ViewState["Order"] as WKF_CASE_ACTIVITY;


            if (_labKit == null)
                _labKit = ViewState["LabKit"] as WKF_CASE_ACTIVITY;
            if (_labKit != null)
                Info.KitReceivedDate = _labKit.SHIPPING_DETAILs.First().RECEIVED_DATE.HasValue ? _labKit.SHIPPING_DETAILs.First().RECEIVED_DATE.Value.ToString("d") : string.Empty;

            BindDataList();

            BioLabResultUserControl1.SetfieldValues(Info, false);

            BreadCrumbLabel.Text = $"My Tasks > Lab Results > {_activitySt.NAME} > Workflow ID {_wCase.WKF_CASE_ID}";
        }
    }

    public override string TopMenuName()
    {
        return "My Tasks";
    }

    public override string MenuItemName()
    {
        return "LabResults";
    }

    public override string AppObjectCode()
    {
        return "LabResults";
    }

    /// <summary>
    /// Database update when "Unaccept" button is clicked.
    /// </summary>
    private void UnacceptButtonClick()
    {
        Deserialize();

        OpenCloseReferralAndWorkflowCase(_wCase, () => "Save");

        Helpers.UnacceptLabResults(_activity, _wCase);

        //(this.Master as RegistryMaster).SetDirtyFlag(false);
        var master = this.Master as RegistryMaster;
        if (master != null)
        {
            master.SetDirtyFlag(false);
        }
        Response.Redirect("LabResults.aspx?OrderType=" + Convert.ToInt16(WorkflowCaseActivityStatus.InProcess));
    }

    protected void btnAccept_Click(object sender, EventArgs e)
    {
        if (btnAccept.Text == "Unaccept")
        {
            UnacceptButtonClick();
        }
        else
        {
            Deserialize();
            _wCase.SetAsChangeTrackingRoot(true);
            //Change the activity status to Accepted
            _activity.STD_WKFACTIVITYSTS_ID = Convert.ToInt16(WorkflowCaseActivityStatus.Accepted);
            if (_activity.SPECIMEN_ANALYSIs.Count() > 0)
            {
                SPECIMEN_ANALYSI specmn = _activity.SPECIMEN_ANALYSIs[0];
                specmn.RESULT_ACKNOWLEDGE_FLAG = true;
                specmn.RESULT_ACKNOWLEDGE_DATE = Helpers.ConvertStringToNullable<DateTime>(DateTime.Today.ToShortDateString());
            }
            //Insert a new activity for Interpretation if none exist
            if (Helpers.GetInterpretationLetterWorkcaseActivity(_wCase.WKF_CASE_ID,
                    (int)WorkflowCaseManager.WorkFlowCaseActivityType.InterpretationDiagnosis) == null)
            {
                WKF_CASE_ACTIVITY InterpreTationActivity = _labResultsManager.CreateNewInterpretationActivity(_wCase.WKF_CASE_ID);
                _wCase.WKF_CASE_ACTIVITies.Add(InterpreTationActivity);
                InterpreTationActivity.SetAsInsertOnSubmit();
            }

            OpenCloseReferralAndWorkflowCase(_wCase, () => "Receive");
            SaveCompleteLabResults();

            _labResultsManager.UpdateResultActivity(_wCase);

            _order.SetAsChangeTrackingRoot();
            _order.STD_WKFACTIVITYSTS_ID = (int)WorkflowCaseActivityStatus.Closed;

            _order.LAB_ORDERs.First().LAB_RETURN_DATE = Helpers.ConvertStringToNullable<DateTime>(((TextBox)BioLabResultUserControl1.FindControl("TextBoxReportDate")).Text);


            _labOrderManager.Update(_order);

            //(this.Master as RegistryMaster).SetDirtyFlag(false);
            var master = this.Master as RegistryMaster;
            if (master != null)
            {
                master.SetDirtyFlag(false);
            }
            BindData(_activity.WKF_CASE_ACTIVITY_ID);
        }
    }
    protected void btnReject_Click(object sender, EventArgs e)
    {
        Deserialize();
        _activity.SetAsChangeTrackingRoot(true);
        //Change the activity status to Voided
        _activity.STD_WKFACTIVITYSTS_ID = Convert.ToInt16(WorkflowCaseActivityStatus.Voided);

        OpenCloseReferralAndWorkflowCase(_wCase, () => "Void");
        SaveCompleteLabResults();
        _labResultsManager.UpdateResultActivity(_wCase);

        //(this.Master as RegistryMaster).SetDirtyFlag(false);
        var master = this.Master as RegistryMaster;
        if (master != null)
        {
            master.SetDirtyFlag(false);
        }
        BindData(_activity.WKF_CASE_ACTIVITY_ID);
    }

    protected void btnSave_Click(object sender, EventArgs e)
    {
        Deserialize();
        _activity.SetAsChangeTrackingRoot(true);
        //Change the activity status to In Process
        if (_activity.STD_WKFACTIVITYSTS_ID == (int)WorkflowCaseActivityStatus.New
            || _activity.STD_WKFACTIVITYSTS_ID == (int)WorkflowCaseActivityStatus.InProcess
            || _activity.STD_WKFACTIVITYSTS_ID == (int)WorkflowCaseActivityStatus.Voided)
        {
            _activity.STD_WKFACTIVITYSTS_ID = Convert.ToInt16(WorkflowCaseActivityStatus.InProcess);
        }


        OpenCloseReferralAndWorkflowCase(_wCase, () => "Save");
        SaveCompleteLabResults();
        _labResultsManager.UpdateResultActivity(_wCase);

        var master = this.Master as RegistryMaster;
        if (master != null)
        {
            master.SetDirtyFlag(false);
        }
        BindData(_activity.WKF_CASE_ACTIVITY_ID);

    }

    protected void btnCancel_Click(object sender, EventArgs e)
    {
        RedirectToReferringPage();
    }

    private void Deserialize()
    {
        _wCase = SerializerUtil.Deserialize<WKF_CASE>((string)ViewState["WCase"]);
        _order = SerializerUtil.Deserialize<WKF_CASE_ACTIVITY>((string)ViewState["Order"]);

        _activity = _wCase.WKF_CASE_ACTIVITies[0];

    }

    private void SaveCompleteLabResults()
    {
        PatientLabInfo info = new PatientLabInfo();
        BioLabResultUserControl1.GetfieldValues(info);

        _activity.CONTACT_NAME = info.LabContact;


        if (_activity.SPECIMEN_ANALYSIs.Count() == 0)
        {
            PrepareSpecimenData();

        }
        if (_activity.SPECIMEN_ANALYSIs.Count() > 0)
        {

            _activity.SPECIMEN_ANALYSIs.First().LAB_UID_NUMBER = info.Accession;
            _activity.SPECIMEN_ANALYSIs.First().SPECIMEN_COLLECTION_DATE = Helpers.ConvertStringToNullable<DateTime>(info.SpecColnDate);
            _activity.SPECIMEN_ANALYSIs.First().TIME_ELAPSED_NUMBER = txtElapsedTime.Text.Trim();
            _activity.SPECIMEN_ANALYSIs.First().TOTAL_VOLUME_NUMBER = txtTotalVolume.Text.Trim();
            _activity.SPECIMEN_ANALYSIs.First().ANALYTE_VOLUME_NUMBER = txtVolOfAnalyteTests.Text.Trim();

            _activity.SPECIMEN_ANALYSIs.First().LAB_REPORT_DATE = Helpers.ToNullable<DateTime>(info.ReportDate);

        }
    }

    private void BindDataList()
    {
        if (_activity == null)
        {
            Deserialize();

        }
        if (_activity.SPECIMEN_ANALYSIs.Count() > 0)
        {
            AnalyteList.DataSource = _activity.SPECIMEN_ANALYSIs[0].ANALYSIS_DETAILs.OrderBy("SortByLabResults");
        }
        else
        {
            AnalyteList.DataSource = new List<SPECIMEN_ANALYSI>();
        }

        AnalyteList.DataBind();
    }


    private void DeleteAnalyte(int analyteId)
    {
        ANALYSIS_DETAIL detail = _activity.SPECIMEN_ANALYSIs.First().ANALYSIS_DETAILs.Where(t => t.ANALYSIS_DETAIL_ID == analyteId).SingleOrDefault();

        _wCase.SetAsChangeTrackingRoot(true);
        detail.SetAsDeleteOnSubmit();
        _labResultsManager.UpdateResultActivity(_wCase);


        GetRefreshLabResults(_activity.WKF_CASE_ACTIVITY_ID);
    }

    private void LoadDataListAnalyte(DataListItem item, int detailId)
    {

        ANALYSIS_DETAIL detail = _activity.SPECIMEN_ANALYSIs.First().ANALYSIS_DETAILs.Where(t => t.ANALYSIS_DETAIL_ID == detailId).SingleOrDefault();

        if (detail != null)
        {

            if (detail.STD_TEST_NAME_ID.HasValue)
            {
                DropDownList testNamesList = item.FindControl("TestNameList") as DropDownList;
                if (testNamesList != null)
                {
                    testNamesList.SelectedIndex = testNamesList.Items.IndexOf(testNamesList.Items.FindByValue(detail.STD_TEST_NAME_ID.Value.ToString()));
                }

            }

            if (detail.STD_ANALYTETYPE_ID.HasValue)
            {
                DropDownList analyteNamesList = item.FindControl("AnalyteNameList") as DropDownList;
                if (analyteNamesList != null)
                {
                    analyteNamesList.SelectedIndex = analyteNamesList.Items.IndexOf(analyteNamesList.Items.FindByValue(detail.STD_ANALYTETYPE_ID.Value.ToString()));
                }

                if (detail.STD_ANALYTETYPE_ID.Value == 15)
                {
                    TextBox otherAnalyteBox = item.FindControl("OtherAnalyteBox") as TextBox;
                    if (otherAnalyteBox != null)
                    {
                        otherAnalyteBox.Text = detail.ANALYTE_NAME;
                    }
                }

            }

            if (detail.STD_ANALYSISMETHOD_ID.HasValue)
            {
                DropDownList analysisMethodsList = item.FindControl("AnalysisMethodList") as DropDownList;
                if (analysisMethodsList != null)
                {
                    analysisMethodsList.SelectedIndex = analysisMethodsList.Items.IndexOf(analysisMethodsList.Items.FindByValue(detail.STD_ANALYSISMETHOD_ID.Value.ToString()));
                }

                if (detail.STD_ANALYSISMETHOD_ID == 7)
                {
                    TextBox otherAnalysisBox = item.FindControl("OtherAnalysisMethodBox") as TextBox;
                    if (otherAnalysisBox != null)
                    {
                        otherAnalysisBox.Text = detail.ANALYSIS_METHOD_NAME;
                    }

                }

            }

            //analysis result 
            TextBox analysisResultBox = item.FindControl("AnalysisResultBox") as TextBox;
            if (analysisResultBox != null)
            {
                analysisResultBox.Text = detail.RESULT_TEXT;
            }


            //result units - this unit label is no longer in use, used dropdown instead 
            //Label unitsLabel = item.FindControl("AnalyteUnitLabel") as Label;
            //unitsLabel.Text = detail.RESULT_UNIT_NUMBER;

            DropDownList resultUnitList = item.FindControl("UnitList") as DropDownList;
            if (detail.RESULT_UNIT_ID.HasValue)
            {
                //DropDownList resultUnitList = item.FindControl("UnitList") as DropDownList;
                //resultUnitList.SelectedIndex = resultUnitList.Items.IndexOf(resultUnitList.Items.FindByValue(detail.STD_UNIT2.NAME));
                if (resultUnitList != null)
                {
                    resultUnitList.SelectedValue = detail.RESULT_UNIT_ID.Value.ToString();
                }
            }

            //new dropdown for lab range units
            if (detail.LAB_REFERENCE_RANGE_UNIT_ID.HasValue)
            {
                DropDownList rangeUnitList = item.FindControl("ReferenceRangeUnitList") as DropDownList;
                //rangeUnitList.SelectedIndex = rangeUnitList.Items.IndexOf(rangeUnitList.Items.FindByValue(detail.STD_UNIT.NAME));
                if (rangeUnitList != null)
                {
                    rangeUnitList.SelectedValue = detail.LAB_REFERENCE_RANGE_UNIT_ID.Value.ToString();
                }
            }

            CheckBox commentCheck = item.FindControl("ResultCommentCheck") as CheckBox;
            TextBox rangeBox = item.FindControl("ReferenceRangeBox") as TextBox;

            if (!String.IsNullOrEmpty(detail.LAB_REFERENCE_RANGE))
            {
                if (rangeBox != null)
                {
                    rangeBox.Text = detail.LAB_REFERENCE_RANGE;
                }
            }
            else
            {
                if (commentCheck != null)
                {
                    commentCheck.Checked = true;
                }
            }

            if (!string.IsNullOrEmpty(detail.OTHER_COMMENT_TEXT))
            {
                if (commentCheck != null)
                {
                    commentCheck.Checked = true;
                }
            }

            CheckBox aboveCheck = item.FindControl("AboveReferenceCheck") as CheckBox;

            if (aboveCheck != null)
            {
                aboveCheck.Checked = detail.LAB_ABOVE_REFERENCE_RANGE.Equals(1);
            }


            TextBox evalNoteBox = item.FindControl("EvaluationNoteBox") as TextBox;
            if (evalNoteBox != null)
            {
                evalNoteBox.Text = detail.EVAL_COMMENT_TEXT;
            }

            //evalNoteBox.Text = HttpUtility.HtmlDecode(detail.EVAL_COMMENT_TEXT);

            TextBox commentBox = item.FindControl("CommentBox") as TextBox;
            if (commentBox != null)
            {
                commentBox.Text = detail.OTHER_COMMENT_TEXT;
            }

            //commentBox.Text = HttpUtility.HtmlDecode(detail.OTHER_COMMENT_TEXT);
        }
    }

    private void PrepareSpecimenData()
    {
        SPECIMEN_ANALYSI analysis = null;
        //Add a new fragment if it does not exist
        if (_activity.FRAGMENT_ANALYSIs == null || _activity.FRAGMENT_ANALYSIs.Count() == 0)
        {
            analysis = AddNewSpecimen();
            analysis.SetAsInsertOnSubmit();
        }
        else
        {
            analysis = _activity.SPECIMEN_ANALYSIs[0];
        }
        //LoadObjectWithControlsData(analysis);
    }


    private SPECIMEN_ANALYSI AddNewSpecimen()
    {
        SPECIMEN_ANALYSI specimen = new SPECIMEN_ANALYSI();
        specimen.WKF_CASE_ACTIVITY_ID = _activity.WKF_CASE_ACTIVITY_ID;
        specimen.LAB_ORDER_ID = _labResultsManager.GetLabOrderId(_activity.WKF_CASE_ACTIVITY_ID);
        specimen.RESULT_ACKNOWLEDGE_FLAG = false;
        _activity.SPECIMEN_ANALYSIs.Add(specimen);
        return specimen;
    }

    private void OpenCloseReferralAndWorkflowCase(WKF_CASE parentCase, Func<string> operation)
    {
        ReferralManager rm = new ReferralManager();

        EFR_REFERRAL parentReferral = rm.getReferralByReferralID(parentCase.REFERRAL_ID.Value);

        parentReferral = SerializerUtil.Deserialize<EFR_REFERRAL>(parentReferral.Serialize());
        parentReferral.SetAsChangeTrackingRoot();

        if (operation() == "Void")
        {
            var items = parentReferral.WKF_CASEs.Except(new List<WKF_CASE> { parentCase }, new WorkflowCaseManager.WorkflowCaseComparer());

            if (items.Count(a => a.STD_WKFCASESTS_ID != (int)WorkFlowCaseStatus.Completed) == 0)

                parentReferral.STD_REFERRALSTS_ID = (int)ReferralStatus.Completed;

            parentReferral.WKF_CASEs.Where(c => c.WKF_CASE_ID == parentCase.WKF_CASE_ID).First().STD_WKFCASESTS_ID = (int)WorkFlowCaseStatus.Completed;

        }
        else
        {
            if (parentReferral.STD_REFERRALSTS_ID != (int)ReferralStatus.InProcess)
                parentReferral.STD_REFERRALSTS_ID = (int)ReferralStatus.InProcess;


            if (parentCase.STD_WKFCASESTS_ID != (int)WorkFlowCaseStatus.InProcess)
                parentReferral.WKF_CASEs.Where(c => c.WKF_CASE_ID == parentCase.WKF_CASE_ID).First().STD_WKFCASESTS_ID = (int)WorkFlowCaseStatus.InProcess;

        }

        SuppressSaveSuccessMessage = true;
        rm.Update(parentReferral);

        SuppressSaveSuccessMessage = false;

    }

    private void UpdateCreateDataListAnalyte(DataListItem item, int detailId, bool insertFlag)
    {

        UnitTestReferenceRangeManager unitRangeManager = new UnitTestReferenceRangeManager();

        if (_activity != null)
        {

            ANALYSIS_DETAIL detail;

            if (insertFlag)
            {
                //create detail
                detail = new ANALYSIS_DETAIL();
                detail.SetAsChangeTrackingRoot(EntityState.New, true);

            }
            else
            {
                //we found detail so just update it 
                detail = _activity.SPECIMEN_ANALYSIs.First().ANALYSIS_DETAILs.Where(t => t.ANALYSIS_DETAIL_ID == detailId).SingleOrDefault();

                detail.SetAsChangeTrackingRoot(true);


            }

            DropDownList testNamesList = item.FindControl("TestNameList") as DropDownList;
            if (testNamesList != null)
            {
                if (int.Parse(testNamesList.SelectedValue) == 0)
                {
                    detail.STD_TEST_NAME_ID = null;
                }
                else
                {
                    detail.STD_TEST_NAME_ID = int.Parse(testNamesList.SelectedValue);
                    detail.TEST_NAME = testNamesList.SelectedItem.Text;
                }

            }


            DropDownList analyteNamesList = item.FindControl("AnalyteNameList") as DropDownList;
            if (analyteNamesList != null)
            {
                if (int.Parse(analyteNamesList.SelectedValue) == 0)
                {
                    detail.STD_ANALYTETYPE_ID = null;

                }
                else
                {
                    detail.STD_ANALYTETYPE_ID = int.Parse(analyteNamesList.SelectedValue);

                    if (int.Parse(analyteNamesList.SelectedValue) == 15)
                    {

                        TextBox otherAnalyteBox = item.FindControl("OtherAnalyteBox") as TextBox;
                        if (otherAnalyteBox != null)
                        {
                            detail.ANALYTE_NAME = otherAnalyteBox.Text;
                        }
                    }
                    else
                    {
                        detail.ANALYTE_NAME = analyteNamesList.SelectedItem.Text;

                    }
                }
            }


            DropDownList analyteMethodList = item.FindControl("AnalysisMethodList") as DropDownList;
            if (analyteMethodList != null)
            {
                if (int.Parse(analyteMethodList.SelectedValue) == 0)
                {
                    detail.STD_ANALYSISMETHOD_ID = null;
                    detail.ANALYSIS_METHOD_NAME = null;

                }
                else
                {
                    if (detail.STD_ANALYSISMETHOD_ID.GetValueOrDefault() != int.Parse(analyteMethodList.SelectedValue))
                    {

                        detail.STD_ANALYSISMETHOD_ID = int.Parse(analyteMethodList.SelectedValue);

                        if (int.Parse(analyteMethodList.SelectedValue) == 7)
                        {

                            TextBox otherMethodBox = item.FindControl("OtherAnalysisMethodBox") as TextBox;
                            if (otherMethodBox != null)
                            {
                                detail.ANALYSIS_METHOD_NAME = otherMethodBox.Text;
                            }
                        }
                        else
                        {
                            detail.ANALYSIS_METHOD_NAME = analyteMethodList.SelectedItem.Text;
                        }
                    }
                }

            }


            //analysis result 
            TextBox analysisResultBox = item.FindControl("AnalysisResultBox") as TextBox;
            if (analysisResultBox != null)
            {
                detail.RESULT_TEXT = analysisResultBox.Text;
            }


            STD_ANALYTE_TEST_REFERENCE_RANGE range = new STD_ANALYTE_TEST_REFERENCE_RANGE();

            if (testNamesList != null && analyteNamesList != null)
            {
                range = unitRangeManager.GetTestReferenceRange(int.Parse(testNamesList.SelectedValue), int.Parse(analyteNamesList.SelectedValue));
            }

            DropDownList resultUnitList = item.FindControl("UnitList") as DropDownList;
            if (resultUnitList != null)
            {
                if (int.Parse(resultUnitList.SelectedValue) == 0)
                {
                    detail.RESULT_UNIT_ID = null;
                }
                else
                {
                    detail.RESULT_UNIT_ID = int.Parse(resultUnitList.SelectedValue);
                }
            }


            TextBox rangeBox = item.FindControl("ReferenceRangeBox") as TextBox;
            if (rangeBox != null)
            {
                detail.LAB_REFERENCE_RANGE = rangeBox.Text;
            }


            DropDownList rangeUnitList = item.FindControl("ReferenceRangeUnitList") as DropDownList;
            if (rangeUnitList != null && rangeBox != null)
            {
                if (int.Parse(rangeUnitList.SelectedValue) == 0 || String.IsNullOrEmpty(rangeBox.Text))
                {
                    detail.LAB_REFERENCE_RANGE_UNIT_ID = null;

                }
                else
                {
                    detail.LAB_REFERENCE_RANGE_UNIT_ID = int.Parse(rangeUnitList.SelectedValue);
                }
            }

            if (range != null)
            {
                detail.TEFSC_REFERENCE_RANGE = range.REFERENCE_RANGE;
                detail.TEFSC_REFERENCE_RANGE_UNIT_ID = range.STD_UNITS_ID;
            }
            CheckBox aboveCheck = item.FindControl("AboveReferenceCheck") as CheckBox;

            if (aboveCheck != null)
            {
                string isChecked = aboveCheck.Checked ? "1" : "0";
                detail.LAB_ABOVE_REFERENCE_RANGE = Helpers.ToNullable<int>(isChecked);
            }


            TextBox evalNoteBox = item.FindControl("EvaluationNoteBox") as TextBox;
            if (evalNoteBox != null)
            {
                detail.EVAL_COMMENT_TEXT = evalNoteBox.Text;
            }

            TextBox commentBox = item.FindControl("CommentBox") as TextBox;
            if (commentBox != null)
            {
                detail.OTHER_COMMENT_TEXT = commentBox.Text;
            }

            if (insertFlag)
            {

                if (_wCase.WKF_CASE_ACTIVITies[0].SPECIMEN_ANALYSIs.Count() > 0)
                {
                    _wCase.WKF_CASE_ACTIVITies[0].SPECIMEN_ANALYSIs[0].ANALYSIS_DETAILs.Add(detail);
                    detail.SetAsInsertOnSubmit();
                }
                else
                {
                    SPECIMEN_ANALYSI analysis = new SPECIMEN_ANALYSI();
                    analysis.SetAsChangeTrackingRoot();
                    analysis.WKF_CASE_ACTIVITY_ID = _activity.WKF_CASE_ACTIVITY_ID;
                    analysis.LAB_ORDER_ID = _labResultsManager.GetLabOrderId(_wCase.WKF_CASE_ACTIVITies[0].WKF_CASE_ACTIVITY_ID);

                    analysis.SetAsInsertOnSubmit(true);

                    analysis.ANALYSIS_DETAILs.Add(detail);
                    _wCase.WKF_CASE_ACTIVITies[0].SPECIMEN_ANALYSIs.Add(analysis);
                }

            }

            _labResultsManager.UpdateResultActivity(_wCase);
            GetRefreshLabResults(_activity.WKF_CASE_ACTIVITY_ID);
        }

    }

}
